/**
 * Order Form Handler
 * Quản lý form tạo đơn hàng với SOLID principles
 */

class OrderFormManager {
    constructor() {
        this.rowIndex = 1;
        this.customerTypeId = null;
        this.selectedWarehouseId = null;
        this.products = [];

        this.initializeElements();
        this.attachEventListeners();
    }

    initializeElements() {
        this.elements = {
            customerSelect: document.getElementById('customerSelect'),
            warehouseSelect: document.getElementById('warehouseSelect'),
            leadIdInput: document.getElementById('leadIdInput'),
            customerInfo: document.getElementById('customerInfo'),
            productTableBody: document.getElementById('productTableBody'),
            addProductBtn: document.getElementById('addProductBtn'),
            totalItems: document.getElementById('totalItems'),
            totalQuantity: document.getElementById('totalQuantity'),
            totalAmount: document.getElementById('totalAmount'),
            grandTotal: document.getElementById('grandTotal'),
        };
    }

    attachEventListeners() {
        this.elements.customerSelect.addEventListener('change', () => this.handleCustomerChange());
        this.elements.warehouseSelect.addEventListener('change', () => this.handleWarehouseChange());
        this.elements.addProductBtn.addEventListener('click', () => this.addProductRow());

        // Attach events to initial row
        // this.attachRowEvents(this.elements.productTableBody.querySelector('.product-row'));
        const existingRows = this.elements.productTableBody.querySelectorAll('.product-row');
        existingRows.forEach(row => {
            this.attachRowEvents(row);
        });
    }

    async handleCustomerChange() {
        const selectedOption = this.elements.customerSelect.selectedOptions[0];
        const customerId = this.elements.customerSelect.value;
        const leadId = selectedOption?.dataset.leadId;

        this.elements.leadIdInput.value = leadId || '';

        if (customerId) {
            await this.fetchAndDisplayCustomerInfo(customerId);
        } else {
            this.hideCustomerInfo();
        }
    }

    async handleWarehouseChange() {
        const warehouseId = this.elements.warehouseSelect.value;

        if (!warehouseId) {
            this.clearAllProducts();
            return;
        }

        this.selectedWarehouseId = warehouseId;
        await this.loadProductsByWarehouse(warehouseId);
        this.updateAllProductSelects();
    }

    async fetchAndDisplayCustomerInfo(customerId) {
        try {
            const data = await this.fetchCustomerInfo(customerId);
            this.displayCustomerInfo(data);
            this.customerTypeId = data.customer.customer_type_id;
            this.updateAllProductPrices();
        } catch (error) {
            console.error('Error fetching customer info:', error);
            this.showError('Không thể tải thông tin khách hàng');
        }
    }

    async fetchCustomerInfo(customerId) {
        const response = await fetch(`/orders/customer-info/${customerId}`);

        if (!response.ok) {
            throw new Error('Failed to fetch customer info');
        }

        return response.json();
    }

    displayCustomerInfo(data) {
        document.getElementById('customerType').textContent = data.customer.customer_type?.name || 'N/A';
        document.getElementById('customerRegion').textContent = data.customer.region?.name || 'N/A';
        document.getElementById('customerOrders').textContent = data.latest_orders.length;
        document.getElementById('customerDebt').textContent = this.formatCurrency(data.total_debt);

        this.elements.customerInfo.style.display = 'block';
    }

    hideCustomerInfo() {
        this.elements.customerInfo.style.display = 'none';
    }

    async loadProductsByWarehouse(warehouseId) {
        try {
            const response = await fetch(`/orders/products-by-warehouse?warehouse_id=${warehouseId}`);

            if (!response.ok) {
                throw new Error('Failed to load products');
            }

            const data = await response.json();
            this.products = data.products;
        } catch (error) {
            console.error('Error loading products:', error);
            this.showError('Không thể tải danh sách sản phẩm');
            this.products = [];
        }
    }

    updateAllProductSelects() {
        const selects = this.elements.productTableBody.querySelectorAll('.product-select');
        selects.forEach(select => this.populateProductSelect(select));
    }

    populateProductSelect(select) {
        const currentValue = select.value;

        // Clear and add default option
        select.innerHTML = '<option value="">-- Chọn sản phẩm --</option>';

        // Add products
        this.products.forEach(product => {
            const option = document.createElement('option');
            option.value = product.id;
            option.textContent = product.display;
            option.dataset.price = product.price;
            option.dataset.priceAgent = product.price_agent;
            option.dataset.priceRetail = product.price_retail;
            option.dataset.quantity = product.quantity;

            select.appendChild(option);
        });

        // Restore selection if valid
        if (currentValue) {
            select.value = currentValue;
        }
    }

    addProductRow() {
        if (!this.selectedWarehouseId) {
            this.showError('Vui lòng chọn kho trước');
            return;
        }

        const tbody = this.elements.productTableBody;
        const newRow = this.createProductRow();

        tbody.appendChild(newRow);
        this.attachRowEvents(newRow);
    }

    createProductRow() {
        const template = this.elements.productTableBody.querySelector('.product-row');
        const newRow = template.cloneNode(true);

        newRow.dataset.index = this.rowIndex;

        // Update input names
        newRow.querySelectorAll('[name]').forEach(input => {
            input.name = input.name.replace(/\[\d+\]/, `[${this.rowIndex}]`);
        });

        // Update data-index for select
        const select = newRow.querySelector('.product-select');
        select.dataset.index = this.rowIndex;

        // Populate products
        this.populateProductSelect(select);

        // Reset values
        newRow.querySelector('.product-select').value = '';
        newRow.querySelector('.unit-price').value = '';
        newRow.querySelector('.quantity').value = '1';
        newRow.querySelector('.line-total').value = '';

        this.rowIndex++;
        return newRow;
    }

    attachRowEvents(row) {
        const productSelect = row.querySelector('.product-select');
        const quantityInput = row.querySelector('.quantity');
        const removeBtn = row.querySelector('.remove-product-btn');

        productSelect.addEventListener('change', () => this.handleProductChange(row));
        quantityInput.addEventListener('input', () => this.calculateLineTotal(row));
        removeBtn.addEventListener('click', () => this.removeProductRow(row));
    }

    handleProductChange(row) {
        this.updateProductPrice(row);
        this.calculateLineTotal(row);
    }

    updateProductPrice(row) {
        const select = row.querySelector('.product-select');
        const priceInput = row.querySelector('.unit-price');
        const selectedOption = select.selectedOptions[0];

        if (!selectedOption || !selectedOption.value) {
            priceInput.value = '';
            return;
        }

        const price = this.getPriceForCustomerType(selectedOption);
        priceInput.value = price;
    }

    getPriceForCustomerType(option) {
        if (!this.customerTypeId) {
            return parseFloat(option.dataset.price) || 0;
        }

        switch (this.customerTypeId) {
            case 1: // Agent
                return parseFloat(option.dataset.priceAgent) || parseFloat(option.dataset.price) || 0;
            case 2: // Retail
                return parseFloat(option.dataset.priceRetail) || parseFloat(option.dataset.price) || 0;
            default:
                return parseFloat(option.dataset.price) || 0;
        }
    }

    calculateLineTotal(row) {
        const price = parseFloat(row.querySelector('.unit-price').value) || 0;
        const quantity = parseInt(row.querySelector('.quantity').value) || 0;
        const lineTotal = price * quantity;

        row.querySelector('.line-total').value = this.formatCurrency(lineTotal);
        this.updateTotals();
    }

    removeProductRow(row) {
        const rows = this.elements.productTableBody.querySelectorAll('.product-row');

        if (rows.length <= 1) {
            this.showError('Phải có ít nhất một sản phẩm');
            return;
        }

        row.remove();
        this.updateTotals();
    }

    updateTotals() {
        const stats = this.calculateOrderStats();

        this.elements.totalItems.textContent = stats.totalItems;
        this.elements.totalQuantity.textContent = stats.totalQuantity;
        this.elements.totalAmount.textContent = this.formatCurrency(stats.grandTotal);
        this.elements.grandTotal.textContent = this.formatCurrency(stats.grandTotal);
    }

    calculateOrderStats() {
        const rows = this.elements.productTableBody.querySelectorAll('.product-row');
        let totalItems = 0;
        let totalQuantity = 0;
        let grandTotal = 0;

        rows.forEach(row => {
            const select = row.querySelector('.product-select');
            if (select.value) {
                totalItems++;
                const quantity = parseInt(row.querySelector('.quantity').value) || 0;
                const price = parseFloat(row.querySelector('.unit-price').value) || 0;

                totalQuantity += quantity;
                grandTotal += price * quantity;
            }
        });

        return { totalItems, totalQuantity, grandTotal };
    }

    updateAllProductPrices() {
        const rows = this.elements.productTableBody.querySelectorAll('.product-row');
        rows.forEach(row => {
            const select = row.querySelector('.product-select');
            if (select.value) {
                this.updateProductPrice(row);
                this.calculateLineTotal(row);
            }
        });
    }

    clearAllProducts() {
        const selects = this.elements.productTableBody.querySelectorAll('.product-select');
        selects.forEach(select => {
            select.innerHTML = '<option value="">-- Chọn kho trước --</option>';
            select.value = '';
        });

        this.products = [];
        this.updateTotals();
    }

    formatCurrency(amount) {
        return new Intl.NumberFormat('vi-VN').format(amount) + ' đ';
    }

    showError(message) {
        alert(message); // Replace with better notification system
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new OrderFormManager();
});