<?php
namespace App\Services;
use App\Models\CRM\ProductStock;
use App\Models\CRM\StockMovement;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
class ProductStockService
{
	/**
	 * Kiểm tra tồn kho
	 */
	public function checkStock($productId, $warehouseId): int
	{
		$stock = ProductStock::where('product_id', $productId)
		                     ->where('warehouse_id', $warehouseId)
		                     ->first();
		return $stock ? $stock->qty : 0;
	}
	/**
	 * Kiểm tra đủ tồn kho cho nhiều sản phẩm
	 */
	public function checkMultipleStock(array $items, $warehouseId): array
	{
		$insufficientItems = [];
		foreach ($items as $item) {
			$currentStock = $this->checkStock($item['product_id'], $warehouseId);
			if ($currentStock < $item['quantity']) {
				$insufficientItems[] = [
					'product_id' => $item['product_id'],
					'required' => $item['quantity'],
					'available' => $currentStock,
					'shortage' => $item['quantity'] - $currentStock,
				];
			}
		}
		return $insufficientItems;
	}
	/**
	 * Điều chỉnh tồn kho
	 */
	public function adjustStock(
		$productId,
		$warehouseId,
		int $changeQty,
		string $reason = null,
		$referenceId = null
	): void {
		DB::transaction(function () use ($productId, $warehouseId, $changeQty, $reason, $referenceId) {
			// Lấy hoặc tạo stock record
			$stock = ProductStock::firstOrCreate(
				[
					'product_id' => $productId,
					'warehouse_id' => $warehouseId,
				],
				[
					'qty' => 0,
				]
			);
			// Cập nhật số lượng
			$stock->increment('qty', $changeQty);
			// Kiểm tra không cho phép âm
			if ($stock->qty < 0) {
				throw new \Exception("Không đủ tồn kho. Hiện tại: {$stock->qty}, Yêu cầu: " . abs($changeQty));
			}
			// Ghi lại stock movement
			StockMovement::create([
				'product_id' => $productId,
				'warehouse_id' => $warehouseId,
				'change_qty' => $changeQty,
				'reason' => $reason,
				'reference_id' => $referenceId,
				'created_by' => Auth::id(),
			]);
		});
	}
	/**
	 * Nhập kho
	 */
	public function stockIn($productId, $warehouseId, int $quantity, string $reason = 'Nhập kho'): void
	{
		$this->adjustStock($productId, $warehouseId, $quantity, $reason);
	}
	/**
	 * Xuất kho
	 */
	public function stockOut($productId, $warehouseId, int $quantity, string $reason = 'Xuất kho', $referenceId = null): void
	{
		$this->adjustStock($productId, $warehouseId, -$quantity, $reason, $referenceId);
	}
	/**
	 * Chuyển kho
	 */
	public function transferStock(
		$productId,
		$fromWarehouseId,
		$toWarehouseId,
		int $quantity,
		string $reason = 'Chuyển kho'
	): void {
		DB::transaction(function () use ($productId, $fromWarehouseId, $toWarehouseId, $quantity, $reason) {
			// Xuất từ kho nguồn
			$this->stockOut($productId, $fromWarehouseId, $quantity, $reason . ' (xuất)');
			// Nhập vào kho đích
			$this->stockIn($productId, $toWarehouseId, $quantity, $reason . ' (nhập)');
		});
	}
	/**
	 * Lấy lịch sử xuất nhập kho
	 */
	public function getStockHistory($productId, $warehouseId = null, $limit = 50): array|LengthAwarePaginator {
		$query = StockMovement::where('product_id', $productId)
		                      ->with(['product', 'warehouse', 'creator']);
		if ($warehouseId) {
			$query->where('warehouse_id', $warehouseId);
		}
		return $query->latest()->paginate($limit);
	}
	/**
	 * Lấy tổng tồn kho theo sản phẩm
	 */
	public function getTotalStockByProduct($productId): int
	{
		return ProductStock::where('product_id', $productId)->sum('qty');
	}
	/**
	 * Lấy danh sách sản phẩm sắp hết hàng
	 */
	public function getLowStockProducts($threshold = 10): Collection|array {
		return ProductStock::with(['product', 'warehouse'])
		                   ->where('qty', '<=', $threshold)
		                   ->where('qty', '>', 0)
		                   ->get();
	}
	/**
	 * Lấy danh sách sản phẩm hết hàng
	 */
	public function getOutOfStockProducts(): Collection|array {
		return ProductStock::with(['product', 'warehouse'])
		                   ->where('qty', '=', 0)
		                   ->get();
	}
	/**
	 * Báo cáo tồn kho theo kho
	 */
	public function getStockReportByWarehouse($warehouseId): Collection|array {
		return ProductStock::where('warehouse_id', $warehouseId)
		                   ->with('product')
		                   ->orderBy('qty', 'asc')
		                   ->get();
	}
	/**
	 * Báo cáo tồn kho tổng hợp
	 */
	public function getStockSummary(): Collection|array {
		return ProductStock::select(
			'product_id',
			DB::raw('SUM(qty) as total_qty'),
			DB::raw('COUNT(warehouse_id) as warehouse_count')
		)
		                   ->with('product')
		                   ->groupBy('product_id')
		                   ->having('total_qty', '>', 0)
		                   ->get();
	}
}