<?php

namespace App\Services;

use App\Models\{MediaFile, Product};
use App\Repositories\Interfaces\{ProductCategoryRepositoryInterface, ProductRepositoryInterface};
use Illuminate\Support\{Arr, Facades\DB};

class ProductService
{
	private const CUSTOMER_TYPE_AGENT = 1;
	private const CUSTOMER_TYPE_RETAIL = 2;

	public function __construct(
		protected ProductRepositoryInterface $repo,
		protected ProductCategoryRepositoryInterface $categories,
		protected StockService $stockService,
		protected WarehouseService $warehouseService
	) {}

	public function list()
	{
		return $this->repo->getAll();
	}

	public function formSelections(): array
	{
		return [
			'categories' => $this->categories->allOrdered(),
			'warehouses' => $this->warehouseService->options(),
		];
	}

	public function create(array $data)
	{
		return DB::transaction(function () use ($data) {
			$product = $this->repo->create($this->productPayload($data));
			$this->syncMedia($product, $data);
			$this->applyInitialStock($product, $data);
			return $product->fresh(['stocks', 'mainImage.media.metadata']);
		});
	}

	public function update(Product $product, array $data)
	{
		return DB::transaction(function () use ($product, $data) {
			$this->repo->update($product, $this->productPayload($data));
			$this->syncMedia($product, $data);
			$this->applyStockAdjustment($product, $data);
			return $product->fresh(['stocks', 'mainImage.media.metadata']);
		});
	}

	public function delete(Product $product)
	{
		return $this->repo->delete($product);
	}

	public function prepareFormData(?Product $product): array
	{
		$selectedWarehouse = old('warehouse_id') ?? $this->getFirstWarehouseId($product);
		$currentQuantity = $this->getCurrentQuantity($product, $selectedWarehouse);
		$defaultQuantity = old('quantity') ?? $this->getFirstStockQuantity($product);

		$existingGallery = $product ? $product->allMedia->where('usage_type', 'gallery') : collect();
		$defaultGalleryIds = old('gallery_ids', $existingGallery->pluck('media_id')->implode(','));

		return [
			'selectedWarehouse' => $selectedWarehouse,
			'currentQuantity' => $currentQuantity,
			'defaultQuantity' => $defaultQuantity,
			'existingGallery' => $existingGallery,
			'defaultGalleryIds' => $defaultGalleryIds,
		];
	}

	/**
	 * Lấy danh sách sản phẩm theo kho (CÁCH 2)
	 */
	public function getProductsForSelect(?int $warehouseId = null)
	{
		$query = Product::query()
		                ->select('crm_product_catalog.id', 'crm_product_catalog.name', 'crm_product_catalog.sku',
			                'price', 'price_agent', 'price_retail');

		if ($warehouseId) {
			$query->leftJoin('crm_product_stock', function ($join) use ($warehouseId) {
				$join->on('crm_product_stock.product_id', '=', 'crm_product_catalog.id')
				     ->where('crm_product_stock.warehouse_id', $warehouseId);
			})
			      ->selectRaw('COALESCE(crm_product_stock.qty, 0) as quantity')
			      ->where('crm_product_stock.qty', '>', 0);
		} else {
			$query->selectRaw('0 as quantity');
		}

		return $query->orderBy('name')
		             ->get()
		             ->map(fn($product) => $this->mapProductForSelect($product));
	}

	public function getPriceForCustomerType(int $productId, ?int $customerTypeId): float
	{
		$product = Product::find($productId);

		if (!$product) {
			return 0;
		}

		return match($customerTypeId) {
			self::CUSTOMER_TYPE_AGENT => $product->price_agent ?? $product->price,
			self::CUSTOMER_TYPE_RETAIL => $product->price_retail ?? $product->price,
			default => $product->price,
		};
	}

	public function find(int $id)
	{
		return Product::findOrFail($id);
	}

	public function search(array $filters = [])
	{
		$query = Product::query();

		$this->applySearchFilter($query, $filters);
		$this->applyCategoryFilter($query, $filters);
		$this->applyStockStatusFilter($query, $filters);

		return $query->orderBy('name')->paginate($filters['per_page'] ?? 20);
	}

	public function getBestSellers(int $limit = 10)
	{
		return Product::query()
		              ->withCount(['orderItems as total_sold' => function ($query) {
			              $query->select(DB::raw('SUM(quantity)'));
		              }])
		              ->orderByDesc('total_sold')
		              ->take($limit)
		              ->get();
	}

	public function getLowStockProducts(int $threshold = 10)
	{
		return Product::query()
		              ->where('quantity', '>', 0)
		              ->where('quantity', '<=', $threshold)
		              ->orderBy('quantity')
		              ->get();
	}

	public function getOutOfStockProducts()
	{
		return Product::where('quantity', '=', 0)->get();
	}

	public function updateQuantity(int $productId, int $quantity)
	{
		return DB::transaction(function () use ($productId, $quantity) {
			$product = $this->find($productId);
			$product->update(['quantity' => $quantity]);
			return $product;
		});
	}

	public function increaseQuantity(int $productId, int $amount)
	{
		return DB::transaction(function () use ($productId, $amount) {
			$product = $this->find($productId);
			$product->increment('quantity', $amount);
			return $product->fresh();
		});
	}

	public function decreaseQuantity(int $productId, int $amount)
	{
		return DB::transaction(function () use ($productId, $amount) {
			$product = $this->find($productId);

			if ($product->quantity < $amount) {
				throw new \Exception("Không đủ tồn kho. Hiện tại: {$product->quantity}, Yêu cầu: {$amount}");
			}

			$product->decrement('quantity', $amount);
			return $product->fresh();
		});
	}

	public function getStatistics(): array
	{
		return [
			'total_products' => Product::count(),
			'in_stock' => Product::where('quantity', '>', 0)->count(),
			'low_stock' => Product::whereBetween('quantity', [1, 10])->count(),
			'out_of_stock' => Product::where('quantity', '=', 0)->count(),
			'total_inventory_value' => Product::sum(DB::raw('price * quantity')),
		];
	}

	// Private Helper Methods
	private function productPayload(array $data): array
	{
		return Arr::only($data, [
			'name', 'sku', 'unit', 'price',
			'price_agent', 'price_retail', 'category_id',
		]);
	}

	private function syncMedia(Product $product, array $data): void
	{
		$this->syncMainImage($product, $data);
		$this->syncGallery($product, $data);
	}

	private function syncMainImage(Product $product, array $data): void
	{
		if (!array_key_exists('main_image_id', $data)) {
			return;
		}

		$product->allMedia()->where('usage_type', 'main_image')->delete();

		if (!empty($data['main_image_id'])) {
			$media = MediaFile::find($data['main_image_id']);
			if ($media) {
				$product->attachMedia($media, 'main_image');
			}
		}
	}

	private function syncGallery(Product $product, array $data): void
	{
		if (!array_key_exists('gallery_ids', $data)) {
			return;
		}

		$product->allMedia()->where('usage_type', 'gallery')->delete();
		$galleryIds = $this->normalizeGalleryIds($data['gallery_ids']);

		if (!empty($galleryIds)) {
			MediaFile::query()
			         ->whereIn('id', $galleryIds)
			         ->get()
			         ->each(fn(MediaFile $media) => $product->attachMedia($media, 'gallery'));
		}
	}

	private function applyInitialStock(Product $product, array $data): void
	{
		$warehouseId = $data['warehouse_id'] ?? null;
		$quantity = (int)($data['quantity'] ?? 0);

		if ($warehouseId && $quantity > 0) {
			$this->stockService->moveStock(
				$product->id,
				$warehouseId,
				$quantity,
				'initial_stock',
				$product->id,
				auth()->id()
			);
		}
	}

	private function applyStockAdjustment(Product $product, array $data): void
	{
		if (!array_key_exists('quantity', $data)) {
			return;
		}

		$warehouseId = $data['warehouse_id'] ?? null;
		if (!$warehouseId) {
			return;
		}

		$desiredQty = (int)($data['quantity'] ?? 0);
		$currentQty = (int)($product->stocks()
		                            ->where('warehouse_id', $warehouseId)
		                            ->value('qty') ?? 0);

		$difference = $desiredQty - $currentQty;

		if ($difference !== 0) {
			$this->stockService->moveStock(
				$product->id,
				$warehouseId,
				$difference,
				'manual_adjustment',
				$product->id,
				auth()->id()
			);
		}
	}

	private function normalizeGalleryIds(null|array|string $galleryIds): array
	{
		if (empty($galleryIds)) {
			return [];
		}

		if (is_array($galleryIds)) {
			return array_filter($galleryIds);
		}

		return array_filter(array_map('trim', explode(',', $galleryIds)));
	}

	private function getFirstWarehouseId(?Product $product): ?int
	{
		if (!$product || $product->stocks->isEmpty()) {
			return null;
		}

		return $product->stocks->first()->warehouse_id;
	}

	private function getCurrentQuantity(?Product $product, ?int $warehouseId): int
	{
		if (!$warehouseId || !$product) {
			return 0;
		}

		return $product->stocks()
		               ->where('warehouse_id', $warehouseId)
		               ->value('qty') ?? 0;
	}

	private function getFirstStockQuantity(?Product $product): ?int
	{
		if (!$product || $product->stocks->isEmpty()) {
			return null;
		}

		return $product->stocks->first()->qty;
	}

	private function mapProductForSelect($product): array
	{
		return [
			'id' => $product->id,
			'name' => $product->name,
			'sku' => $product->sku,
			'display' => "{$product->name} ({$product->sku})",
			'price' => $product->price,
			'price_agent' => $product->price_agent,
			'price_retail' => $product->price_retail,
			'quantity' => (int)$product->quantity,
		];
	}

	private function applySearchFilter($query, array $filters): void
	{
		if (empty($filters['search'])) {
			return;
		}

		$search = $filters['search'];
		$query->where(function ($q) use ($search) {
			$q->where('name', 'like', "%{$search}%")
			  ->orWhere('sku', 'like', "%{$search}%");
		});
	}

	private function applyCategoryFilter($query, array $filters): void
	{
		if (!empty($filters['category_id'])) {
			$query->where('category_id', $filters['category_id']);
		}
	}

	private function applyStockStatusFilter($query, array $filters): void
	{
		if (!isset($filters['stock_status'])) {
			return;
		}

		match($filters['stock_status']) {
			'in_stock' => $query->where('quantity', '>', 0),
			'low_stock' => $query->whereBetween('quantity', [1, 10]),
			'out_of_stock' => $query->where('quantity', '=', 0),
			default => null,
		};
	}
}