<?php

namespace App\Services;

use App\Models\CRM\CustomerDebt;
use App\Models\CRM\Order;
use App\Models\CRM\OrderApproval;
use App\Models\CRM\OrderStatusHistory;
use App\Models\CRM\OrderStatusType;
use App\Models\CRM\Payment;
use App\Models\OrderNotification;
use App\Repositories\Interfaces\OrderRepositoryInterface;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderService
{
    private const DEPARTMENT_FLOW = [
        'sales' => 'ketoan',
        'ketoan' => 'duyet1',
        'duyet1' => 'duyet2',
        'duyet2' => 'kho',
        'kho' => 'completed',
    ];

    private const STATUS_CODES = [
        'sales' => 'PENDING_KETOAN',
        'ketoan' => 'PENDING_DUYET1',
        'duyet1' => 'PENDING_DUYET2',
        'duyet2' => 'APPROVED_DUYET2',
        'kho' => 'COMPLETED',
    ];

    private const DEPARTMENT_NAMES = [
        'sales' => 'Sales',
        'ketoan' => 'Kế toán',
        'duyet1' => 'Duyệt cấp 1',
        'duyet2' => 'Giám đốc',
        'kho' => 'Kho',
    ];

    public function __construct(
        protected OrderRepositoryInterface $orderRepo,
        // Đảm bảo bạn đã tạo 2 file Service này hoặc xóa dòng này nếu chưa có
        protected ProductStockService $stockService,
        protected NotificationService $notificationService
    ) {}

    public function count(): int
    {
        return $this->orderRepo->count();
    }

    public function getRecentOrders($limit = 5): Collection {
        return $this->orderRepo->getRecent($limit);
    }

    public function getOrdersByUserRole($user, array $filters = []): LengthAwarePaginator
    {
        if ($user->hasRole('sales')) {
            return $this->orderRepo->getOrdersBySales($user->id, $filters);
        }
        if ($user->hasRole('ketoan')) {
            return $this->orderRepo->getOrdersByDepartment('ketoan', $filters);
        }
        if ($user->hasRole('duyet1') || $user->hasRole('duyet2')) {
            $level = $user->hasRole('duyet1') ? 'duyet1' : 'duyet2';
            return $this->orderRepo->getOrdersByDepartment($level, $filters);
        }
        if ($user->hasRole('kho')) {
            return $this->orderRepo->getOrdersByDepartment('kho', $filters);
        }
        return $this->orderRepo->search($filters);
    }

    /**
     * Tạo đơn hàng mới
     */
    public function createOrder(array $data): Order
    {
        return DB::transaction(function () use ($data) {
            $order = $this->createOrderRecord($data);
            $this->createOrderItems($order, $data['items']);
            $this->createInitialApproval($order);
            // Fix: Truyền đúng key status code
            $this->recordStatusHistory($order, null, 'sales', 'Đơn hàng được tạo bởi Sales');

            return $order->fresh(['items.product', 'lead.customer']);
        });
    }

    /**
     * Cập nhật đơn hàng (chỉ khi chưa duyệt)
     */
    public function updateOrder($id, array $data): Order
    {
        return DB::transaction(function () use ($id, $data) {
            $order = $this->orderRepo->find($id);

            if ($order->current_department !== 'sales') {
                throw new \Exception('Không thể sửa đơn đã gửi duyệt');
            }

            // 1. Update thông tin chung
            $this->updateOrderRecord($order, $data);

            // 2. Update sản phẩm (Logic quan trọng để fix lỗi xóa)
            if (isset($data['items'])) {
                $this->updateOrderItems($order, $data['items']);
            }

            return $order->fresh(['items.product']);
        });
    }

    /**
     * Gửi đơn lên duyệt
     */
    public function submitForApproval($id): void
    {
        DB::transaction(function () use ($id) {
            $order = $this->orderRepo->find($id);
            if ($order->current_department !== 'sales') {
                throw new \Exception('Đơn hàng đã được gửi duyệt');
            }
            $this->transitionOrderToDepartment($order, 'ketoan', 'PENDING_KETOAN');
            $this->createApprovalRecord($order, 'ketoan');
            $this->recordStatusHistory($order, 'sales', 'ketoan', 'Sales gửi đơn lên duyệt');
            $this->notificationService->notifyDepartment('ketoan', $order, 'Đơn hàng mới cần kiểm tra');
        });
    }

    public function processApproval(string $id, array $data): void
    {
        if ($data['action'] === 'approve') {
            $this->approveOrder($id, $data);
        } else {
            $this->rejectOrder($id, $data);
        }
    }

    /**
     * Duyệt đơn hàng
     */
    public function approveOrder($id, array $data): void
    {
        DB::transaction(function () use ($id, $data) {
            $order = $this->orderRepo->find($id);
            $user = Auth::user();
            $currentDept = $order->current_department;

            $nextDept = $this->getNextDepartment($currentDept);

            // Cập nhật log duyệt hiện tại
            $this->updateCurrentApproval($order, $currentDept, $user, $data);

            // Chuyển trạng thái đơn hàng
            $this->transitionOrderToDepartment($order, $nextDept, $this->getNextStatusCode($currentDept));

            // Xử lý đặc biệt cho cấp duyệt cuối
            if ($currentDept === 'duyet2') {
                $this->finalizeApproval($order, $user);
            }

            // Tạo log chờ duyệt cho cấp tiếp theo
            if ($nextDept !== 'completed') {
                $this->createApprovalRecord($order, $nextDept);
            }

            $this->recordApprovalHistory($order, $currentDept, $nextDept, $user, $data);
            $this->sendApprovalNotifications($order, $currentDept, $nextDept);
        });
    }

    /**
     * Từ chối đơn hàng
     */
    public function rejectOrder($id, array $data): void
    {
        DB::transaction(function () use ($id, $data) {
            $order = $this->orderRepo->find($id);
            $user = Auth::user();
            $currentDept = $order->current_department;

            $this->updateRejectedApproval($order, $currentDept, $user, $data);

            // Quay về Sales
            $this->transitionOrderToDepartment($order, 'sales', 'REJECTED'); // Đảm bảo code REJECTED tồn tại trong DB

            $this->recordRejectionHistory($order, $currentDept, $user, $data);
            $this->notifyRejection($order, $currentDept, $data['rejection_reason']);
        });
    }

    /**
     * Xuất kho
     */
    public function shipOrder($id, array $data): void
    {
        DB::transaction(function () use ($id, $data) {
            $order = $this->orderRepo->find($id);

            if ($order->current_department !== 'kho') {
                throw new \Exception('Đơn hàng chưa được duyệt đầy đủ');
            }

            $this->adjustInventory($order);
            $this->transitionOrderToDepartment($order, 'completed', 'COMPLETED');
            $this->updateWarehouseApproval($order);
            $this->recordShippingHistory($order, $data);
            $this->createDebtIfNeeded($order);
            $this->notifyShipping($order);
        });
    }

    /**
     * Ghi nhận thanh toán
     */
    public function recordPayment($id, array $data): void
    {
        DB::transaction(function () use ($id, $data) {
            $order = $this->orderRepo->find($id);
            $this->createPaymentRecord($order, $data);
            $this->updateDebtRecord($order, $data['amount']);
            $this->checkPaymentCompletion($order);
            $this->notifyPayment($order, $data['amount']);
        });
    }

    /**
     * Hủy đơn hàng
     */
    public function cancelOrder($id): void
    {
        DB::transaction(function () use ($id) {
            $order = $this->orderRepo->find($id);

            if ($order->current_department === 'completed') {
                throw new \Exception('Không thể hủy đơn đã xuất kho');
            }

            $this->transitionOrderToDepartment($order, 'sales', 'REJECTED'); // Hoặc CANCELLED tùy DB
            $this->recordStatusHistory($order, $order->current_department, 'sales', 'Đơn hàng bị hủy');
        });
    }

    // ==================== HELPER METHODS ====================

    private function createOrderRecord(array $data): Order
    {
        // Fix: Đảm bảo code này tồn tại trong Seeder
        $initialStatus = OrderStatusType::where('code', 'PENDING_KETOAN')->first(); // Hoặc PENDING_APPROVAL

        return $this->orderRepo->create([
            'order_code' => $this->generateOrderCode(),
            'lead_id' => $data['lead_id'],
            'order_date' => $data['order_date'],
            'warehouse_id' => $data['warehouse_id'],
            'total_amount' => $this->calculateTotalAmount($data['items']),
            'created_by' => Auth::id(),
            'current_department' => 'sales',
            'current_status_type_id' => $initialStatus ? $initialStatus->id : null,
        ]);
    }

    private function createOrderItems(Order $order, array $items): void
    {
        foreach ($items as $item) {
            $order->items()->create($item);
        }
    }

    private function updateOrderRecord(Order $order, array $data): void
    {
        $updateData = [
            'lead_id' => $data['lead_id'] ?? $order->lead_id,
            'order_date' => $data['order_date'] ?? $order->order_date,
            'warehouse_id' => $data['warehouse_id'] ?? $order->warehouse_id,
        ];
        $order->update($updateData);
    }

    /**
     * FIX LỖI UPDATE: Logic đồng bộ thông minh
     * Thay vì xóa hết tạo lại, ta sẽ check ID để giữ lại cái cũ, xóa cái thừa
     */
    private function updateOrderItems(Order $order, array $items): void
    {
        // 1. Lấy danh sách ID sản phẩm được gửi lên
        $submittedProductIds = collect($items)->pluck('product_id')->filter()->toArray();

        // 2. XÓA: Các item trong DB mà không có trong danh sách gửi lên
        $order->items()->whereNotIn('product_id', $submittedProductIds)->delete();

        // 3. THÊM / CẬP NHẬT
        $totalAmount = 0;
        foreach ($items as $item) {
            // Sử dụng updateOrCreate để giữ nguyên ID của item cũ (nếu cần)
            $order->items()->updateOrCreate(
                [
                    'order_id' => $order->id,
                    'product_id' => $item['product_id']
                ],
                [
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'line_total' => $item['quantity'] * $item['unit_price']
                ]
            );
            $totalAmount += ($item['quantity'] * $item['unit_price']);
        }

        // Cập nhật lại tổng tiền đơn hàng
        $order->update(['total_amount' => $totalAmount]);
    }

    private function createInitialApproval(Order $order): void
    {
        OrderApproval::create([
            'order_id' => $order->id,
            'level' => 'sales',
            'status' => 'approved',
            'approved_by' => Auth::id(),
            'approved_at' => now(),
        ]);
    }

    private function createApprovalRecord(Order $order, string $level): void
    {
        OrderApproval::firstOrCreate([
            'order_id' => $order->id,
            'level' => $level,
            'status' => 'pending',
        ]);
    }

    private function transitionOrderToDepartment(Order $order, string $dept, string $statusCode): void
    {
        $status = OrderStatusType::where('code', $statusCode)->first();

        $order->update([
            'current_department' => $dept,
            'current_status_type_id' => $status ? $status->id : $order->current_status_type_id,
        ]);
    }

    private function updateCurrentApproval(Order $order, string $dept, $user, array $data): void
    {
        $approval = OrderApproval::where('order_id', $order->id)
            ->where('level', $dept)
            ->where('status', 'pending')
            ->first();

        // Nếu không tìm thấy (ví dụ admin force approve), có thể tạo mới
        if (!$approval) {
            $approval = new OrderApproval(['order_id' => $order->id, 'level' => $dept]);
        }

        $updateData = [
            'status' => 'approved',
            'approved_by' => $user->id,
            'approved_at' => now(),
            'note' => $data['note'] ?? null,
        ];

        if ($dept === 'ketoan' && isset($data['debt_checked'])) {
            $updateData['meta_data'] = json_encode(['debt_checked' => true, 'debt_note' => $data['debt_note'] ?? null]);
        }

        $approval->fill($updateData)->save();
    }

    private function updateRejectedApproval(Order $order, string $dept, $user, array $data): void
    {
        OrderApproval::updateOrCreate(
            ['order_id' => $order->id, 'level' => $dept, 'status' => 'pending'],
            [
                'status' => 'rejected',
                'approved_by' => $user->id,
                'approved_at' => now(), // rejected cũng cần thời điểm
                'rejection_reason' => $data['rejection_reason'],
                'note' => $data['note'] ?? null
            ]
        );
    }

    private function finalizeApproval(Order $order, $user): void
    {
        $order->update([
            'approved_by' => $user->id,
            'approved_at' => now(),
            'estimated_delivery' => Carbon::now()->addDays(3),
        ]);
    }

    private function adjustInventory(Order $order): void
    {
        foreach ($order->items as $item) {
            $this->stockService->adjustStock(
                $item->product_id,
                $order->warehouse_id,
                -$item->quantity, // Trừ kho
                "Xuất theo đơn hàng #{$order->order_code}",
                $order->id
            );
        }
    }

    private function updateWarehouseApproval(Order $order): void
    {
        OrderApproval::where('order_id', $order->id)
            ->where('level', 'kho')
            ->update([
                'status' => 'approved',
                'approved_by' => Auth::id(),
                'approved_at' => now(),
            ]);
    }

    private function createDebtIfNeeded(Order $order): void
    {
        $totalPaid = $order->payments()->sum('amount');

        if ($totalPaid < $order->total_amount) {
            CustomerDebt::create([
                'customer_id' => $order->lead->customer_id,
                'order_id' => $order->id,
                'total_amount' => $order->total_amount,
                'paid_amount' => $totalPaid,
                'debt_amount' => $order->total_amount - $totalPaid, // Thêm cột này nếu DB có
                'due_date' => Carbon::now()->addDays(30),
                'status' => $totalPaid > 0 ? 'partial' : 'unpaid',
            ]);
        }
    }

    private function createPaymentRecord(Order $order, array $data): void
    {
        Payment::create([
            'order_id' => $order->id,
            'payment_date' => $data['payment_date'],
            'amount' => $data['amount'],
            'method_id' => $data['method_id'],
            'recorded_by' => Auth::id(),
        ]);
    }

    private function updateDebtRecord(Order $order, float $amount): void
    {
        $debt = CustomerDebt::where('order_id', $order->id)->first();

        if ($debt) {
            $debt->paid_amount += $amount;
            $debt->debt_amount = $debt->total_amount - $debt->paid_amount;

            if ($debt->paid_amount >= $debt->total_amount) {
                $debt->status = 'paid';
            } elseif ($debt->paid_amount > 0) {
                $debt->status = 'partial';
            }

            $debt->save();
        }
    }

    private function checkPaymentCompletion(Order $order): void
    {
        $totalPaid = $order->payments()->sum('amount');
        if ($totalPaid >= $order->total_amount) {
            $order->update(['payment_recorded' => true]);
        }
    }

    private function calculateTotalAmount(array $items): float
    {
        return array_reduce($items, function ($total, $item) {
            return $total + ($item['quantity'] * $item['unit_price']);
        }, 0);
    }

    protected function generateOrderCode(): string
    {
        $prefix = 'ORD';
        $date = date('Ymd');
        // Fix: dùng created_at thay vì order_date để count chính xác hơn theo ngày hệ thống
        $count = Order::whereDate('created_at', today())->count() + 1;
        return sprintf('%s%s%04d', $prefix, $date, $count);
    }

    private function getNextDepartment(string $current): string
    {
        return self::DEPARTMENT_FLOW[$current] ?? 'completed';
    }

    private function getNextStatusCode(string $dept): string
    {
        return self::STATUS_CODES[$dept] ?? 'COMPLETED';
    }

    private function getDepartmentName(string $dept): string
    {
        return self::DEPARTMENT_NAMES[$dept] ?? $dept;
    }

    // Fix: recordStatusHistory cần linh hoạt hơn trong việc tìm status_id
    protected function recordStatusHistory(Order $order, ?string $from, string $to, string $note): void
    {
        // Lấy code dựa trên mapping
        $statusCode = self::STATUS_CODES[$to] ?? null;

        if ($to === 'sales' && !$statusCode) $statusCode = 'REJECTED'; // Fallback nếu quay về sales

        $statusType = null;
        if ($statusCode) {
            $statusType = OrderStatusType::where('code', $statusCode)->first();
        }

        OrderStatusHistory::create([
            'order_id' => $order->id,
            'previous_status_id' => $order->getOriginal('current_status_type_id'), // Lưu lại trạng thái cũ
            'new_status_id' => $statusType?->id,
            'from_department' => $from, // Nếu DB không có cột này thì bỏ đi
            'to_department' => $to,     // Nếu DB không có cột này thì bỏ đi
            'note' => $note,
            'changed_by' => Auth::id(),
            'changed_at' => now(),
        ]);
    }

    private function recordApprovalHistory(Order $order, string $from, string $to, $user, array $data): void
    {
        $note = "Đã duyệt bởi {$user->name}";
        if (isset($data['note'])) {
            $note .= " - Ghi chú: {$data['note']}";
        }
        $this->recordStatusHistory($order, $from, $to, $note);
    }

    private function recordRejectionHistory(Order $order, string $dept, $user, array $data): void
    {
        $note = "Bị từ chối bởi {$user->name}: " . ($data['rejection_reason'] ?? '');
        $this->recordStatusHistory($order, $dept, 'sales', $note);
    }

    private function recordShippingHistory(Order $order, array $data): void
    {
        $note = 'Đã xuất kho';
        if (isset($data['shipping_note'])) {
            $note .= " - {$data['shipping_note']}";
        }
        $this->recordStatusHistory($order, 'kho', 'completed', $note);
    }

    private function sendApprovalNotifications(Order $order, string $from, string $to): void
    {
        $this->notificationService->notifySales(
            $order->created_by,
            $order,
            'status_change',
            "Đơn hàng #{$order->order_code} đã được {$this->getDepartmentName($from)} duyệt"
        );

        if ($to !== 'completed') {
            $this->notificationService->notifyDepartment(
                $to,
                $order,
                "Đơn hàng #{$order->order_code} cần xử lý"
            );
        }
    }

    private function notifyRejection(Order $order, string $dept, string $reason): void
    {
        $this->notificationService->notifySales(
            $order->created_by,
            $order,
            'rejected',
            "Đơn hàng #{$order->order_code} bị từ chối bởi {$this->getDepartmentName($dept)}: {$reason}"
        );
    }

    private function notifyShipping(Order $order): void
    {
        $this->notificationService->notifySales(
            $order->created_by,
            $order,
            'shipped',
            "Đơn hàng #{$order->order_code} đã được xuất kho"
        );
    }

    private function notifyPayment(Order $order, float $amount): void
    {
        $formattedAmount = number_format($amount);
        $this->notificationService->notifySales(
            $order->created_by,
            $order,
            'payment_received',
            "Đã ghi nhận thanh toán {$formattedAmount} VNĐ cho đơn #{$order->order_code}"
        );
    }

    // ==================== QUERY METHODS ====================

    public function getOrdersBySales($salesId, array $filters = []): LengthAwarePaginator
    {
        return $this->orderRepo->getOrdersBySales($salesId, $filters);
    }

    public function getOrdersForKeToan(array $filters = []): LengthAwarePaginator
    {
        return $this->orderRepo->getOrdersByDepartment('ketoan', $filters);
    }

    public function getOrdersForApprover($user, array $filters = []): LengthAwarePaginator
    {
        $level = $user->hasRole('duyet1') ? 'duyet1' : 'duyet2';
        return $this->orderRepo->getOrdersByDepartment($level, $filters);
    }

    public function getOrdersForWarehouse(array $filters = []): LengthAwarePaginator
    {
        return $this->orderRepo->getOrdersByDepartment('kho', $filters);
    }

    public function getAllOrders(array $filters = []): LengthAwarePaginator
    {
        return $this->orderRepo->search($filters);
    }

    public function findWithDetails($id): ?Order
    {
        return $this->orderRepo->findWithDetails($id);
    }

    public function find($id)
    {
        return $this->orderRepo->find($id);
    }

    public function getOrderTimeline($orderId)
    {
        return OrderStatusHistory::where('order_id', $orderId)
            ->with('changedBy', 'statusType') // Check lại tên quan hệ trong Model OrderStatusHistory
            ->orderBy('changed_at', 'asc')
            ->get();
    }

    public function getOrderNotifications($orderId)
    {
        $order = $this->orderRepo->find($orderId);

        return OrderNotification::where('order_id', $orderId)
            ->where('user_id', $order->created_by)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function getSalesStatistics($salesId): array
    {
        return [
            'total_orders' => Order::where('created_by', $salesId)->count(),
            'pending' => Order::where('created_by', $salesId)
                ->where('current_department', '!=', 'completed')->count(),
            'completed' => Order::where('created_by', $salesId)
                ->where('current_department', 'completed')->count(),
            'total_revenue' => Order::where('created_by', $salesId)
                ->where('payment_recorded', true)->sum('total_amount'),
        ];
    }

    public function getRecentOrdersBySales($salesId, $limit = 10)
    {
        return Order::where('created_by', $salesId)
            ->with(['lead.customer', 'warehouse', 'currentStatusType'])
            ->latest()
            ->take($limit)
            ->get();
    }

    public function getPendingNotifications($salesId)
    {
        return OrderNotification::where('user_id', $salesId)
            ->where('is_read', false)
            ->with('order')
            ->latest()
            ->get();
    }

    public function getCurrentApprovalLevel(Order $order): ?string
    {
        return $order->current_department;
    }
}