<?php
namespace App\Services;
use App\Models\CRM\Order;
use App\Models\OrderNotification;
use App\Models\User;
use Illuminate\Support\Facades\DB;
class NotificationService
{
	/**
	 * Gửi thông báo cho Sales
	 */
	public function notifySales($salesId, Order $order, string $type, string $message): void
	{
		$titles = [
			'status_change' => '📋 Cập nhật trạng thái đơn hàng',
			'approved' => '✅ Đơn hàng đã được duyệt',
			'rejected' => '❌ Đơn hàng bị từ chối',
			'shipped' => '📦 Đơn hàng đã xuất kho',
			'payment_received' => '💰 Đã ghi nhận thanh toán',
		];
		OrderNotification::create([
			'order_id' => $order->id,
			'user_id' => $salesId,
			'type' => $type,
			'title' => $titles[$type] ?? 'Thông báo',
			'message' => $message,
			'is_read' => false,
		]);
	}
	/**
	 * Gửi thông báo cho bộ phận
	 */
	public function notifyDepartment(string $department, Order $order, string $message): void
	{
		// Lấy danh sách users thuộc department
		$users = $this->getUsersByDepartment($department);
		foreach ($users as $user) {
			OrderNotification::create([
				'order_id' => $order->id,
				'user_id' => $user->id,
				'type' => 'department_notification',
				'title' => '🔔 Đơn hàng mới cần xử lý',
				'message' => $message,
				'is_read' => false,
			]);
		}
	}
	/**
	 * Gửi thông báo quá hạn
	 */
	public function notifyOverdue(Order $order): void
	{
		$salesId = $order->created_by;
		$message = "Đơn hàng #{$order->order_code} đã quá thời gian dự kiến xử lý. " .
		           "Hiện đang ở bộ phận: " . $this->getDepartmentName($order->current_department);
		$this->notifySales($salesId, $order, 'status_change', $message);
	}
	/**
	 * Đánh dấu đã đọc
	 */
	public function markAsRead($notificationId): void
	{
		OrderNotification::where('id', $notificationId)->update([
			'is_read' => true,
			'read_at' => now(),
		]);
	}
	/**
	 * Đánh dấu tất cả đã đọc
	 */
	public function markAllAsRead($userId): void
	{
		OrderNotification::where('user_id', $userId)
		                 ->where('is_read', false)
		                 ->update([
			                 'is_read' => true,
			                 'read_at' => now(),
		                 ]);
	}
	/**
	 * Lấy số thông báo chưa đọc
	 */
	public function getUnreadCount($userId): int
	{
		return OrderNotification::where('user_id', $userId)
		                        ->where('is_read', false)
		                        ->count();
	}
	/**
	 * Lấy danh sách thông báo chưa đọc
	 */
	public function getUnreadNotifications($userId, $limit = 10)
	{
		return OrderNotification::where('user_id', $userId)
		                        ->where('is_read', false)
		                        ->with('order')
		                        ->latest()
		                        ->take($limit)
		                        ->get();
	}
	/**
	 * Lấy tất cả thông báo
	 */
	public function getAllNotifications($userId, $limit = 50)
	{
		return OrderNotification::where('user_id', $userId)
		                        ->with('order')
		                        ->latest()
		                        ->paginate($limit);
	}
	// ==================== HELPER METHODS ====================
	protected function getUsersByDepartment(string $department): \Illuminate\Database\Eloquent\Collection
	{
		// Map department sang role
		$roleMap = [
			'ketoan' => 'ketoan',
			'duyet1' => 'duyet1',
			'duyet2' => 'duyet2',
			'kho' => 'kho',
		];
		$roleName = $roleMap[$department] ?? null;
		if (!$roleName) {
			return collect();
		}
		return User::role($roleName)->get();
	}
	protected function getDepartmentName(string $dept): string
	{
		$names = [
			'sales' => 'Sales',
			'ketoan' => 'Kế toán',
			'duyet1' => 'Duyệt cấp 1',
			'duyet2' => 'Giám đốc',
			'kho' => 'Kho',
			'completed' => 'Hoàn tất',
		];
		return $names[$dept] ?? $dept;
	}
}