<?php
namespace App\Services;
use App\Models\CRM\Customer;
use App\Repositories\Interfaces\CustomerRepositoryInterface;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
class CustomerService
{
	protected CustomerRepositoryInterface $customerRepo;
	public function __construct(CustomerRepositoryInterface $customerRepo)
	{
		$this->customerRepo = $customerRepo;
	}
	/**
	 * Lấy danh sách nâng cao (có relations)
	 * @return mixed
	 */
	public function getAll(): mixed {
		// Gọi method getAllWithRelations() từ repository
		return $this->customerRepo->getAllWithRelations();
	}
	/**
	 * Lấy danh sách khách hàng cho dropdown select (có lead)
	 */
	public function getCustomersForSelect()
	{
		return Customer::query()
			->whereHas('leads')
			->select('crm_customers.*')
			// Subquery để lấy latest lead_id
			->addSelect([
				'latest_lead_id' => \App\Models\CRM\Lead::select('id')
				                                        ->whereColumn('customer_id', 'crm_customers.id')
				                                        ->latest('created_at')
				                                        ->limit(1)
			])
			->with([
				'customerType:id,name',
				'region:id,name'
			])
			->orderBy('name')
			->get()
			->map(function($customer) {
				return [
					'id' => $customer->id,
					'name' => $customer->name,
					'phone' => $customer->phone,
					'display' => $customer->name . ' - ' . $customer->phone,
					'lead_id' => $customer->latest_lead_id, // Từ subquery
					'customer_type_id' => $customer->customer_type_id,
					'customer_type_name' => $customer->customerType->name ?? 'N/A',
					'region_name' => $customer->region->name ?? 'N/A',
				];
			});
	}
	public function count()
	{
		return $this->customerRepo->count();
	}
	public function countPurchased()
	{
		return $this->customerRepo->countPurchased();
	}
	public function find($id)
	{
		return $this->customerRepo->find($id);
	}
	public function findWithDetails($id)
	{
		return $this->customerRepo->findWithDetails($id);
	}
	public function create(array $data)
	{
		return DB::transaction(function () use ($data) {
			// Tự động gán created_by nếu chưa có
			if (!isset($data['created_by']) && auth()->check()) {
				$data['created_by'] = auth()->id();
			}
			// Mặc định là lead nếu chưa chỉ định
			if (!isset($data['customer_status'])) {
				$data['customer_status'] = 'lead';
			}
			return $this->customerRepo->create($data);
		});
	}
	public function search(array $filters = [])
	{
		return $this->customerRepo->search($filters);
	}
	public function update($id, array $data)
	{
		return DB::transaction(fn() => $this->customerRepo->update($id, $data));
	}
	public function delete($id)
	{
		return DB::transaction(fn() => $this->customerRepo->delete($id));
	}
	// ==================== BUSINESS LOGIC ====================

	/**
	 * Chuyển lead thành member
	 */
	public function convertToMember($customerId, array $membershipData = [])
	{
		return DB::transaction(function () use ($customerId, $membershipData) {
			$customer = $this->customerRepo->find($customerId);

			$customer->update([
				'customer_status' => 'member',
				'converted_to_member_at' => now(),
			]);

			// Tạo membership nếu có data
			if (!empty($membershipData)) {
				$customer->memberships()->create($membershipData);
			}

			return $customer;
		});
	}
	/**
	 * Đánh dấu khách hàng tiềm năng
	 */
	public function markAsPotential($customerId, bool $isPotential = true)
	{
		return DB::transaction(function () use ($customerId, $isPotential) {
			$customer = $this->customerRepo->find($customerId);
			$customer->update(['is_potential' => $isPotential]);
			return $customer;
		});
	}
	/**
	 * Gán người phụ trách
	 */
	public function assignOwner($customerId, $ownerId)
	{
		return DB::transaction(function () use ($customerId, $ownerId) {
			$customer = $this->customerRepo->find($customerId);
			$customer->update(['owner_id' => $ownerId]);
			return $customer;
		});
	}
	/**
	 * Thêm tags cho khách hàng
	 */
	public function attachTags($customerId, array $tagIds)
	{
		$customer = $this->customerRepo->find($customerId);
		$customer->tags()->syncWithoutDetaching($tagIds);
		return $customer;
	}

	/**
	 * Xóa tags khỏi khách hàng
	 */
	public function detachTags($customerId, array $tagIds)
	{
		$customer = $this->customerRepo->find($customerId);
		$customer->tags()->detach($tagIds);
		return $customer;
	}

	/**
	 * Đồng bộ tags (thay thế hoàn toàn)
	 */
	public function syncTags($customerId, array $tagIds)
	{
		$customer = $this->customerRepo->find($customerId);
		$customer->tags()->sync($tagIds);
		return $customer;
	}

	/**
	 * Lấy thống kê khách hàng
	 */
	public function getStatistics()
	{
		return [
			'total_customers' => $this->count(),
			'total_purchased' => $this->countPurchased(),
			'total_leads' => Customer::where('customer_status', 'lead')->count(),
			'total_members' => Customer::where('customer_status', 'member')->count(),
			'total_potential' => Customer::where('is_potential', true)->count(),
		];
	}

	/**
	 * Lấy khách hàng có sinh nhật trong tháng
	 */
	public function getBirthdaysThisMonth(): Collection|array {
		// Nếu có trường birthday trong customers table
		return Customer::query()
		                               ->whereMonth('birthday', date('m'))
		                               ->orderByRaw('DAY(birthday)')
		                               ->get();
	}

	/**
	 * Lấy khách hàng chưa tương tác trong X ngày
	 */
	public function getInactiveCustomers($days = 30): Collection|array {
		return Customer::query()
		                               ->whereDoesntHave('interactions', function ($q) use ($days) {
			                               $q->where('interaction_date', '>=', now()->subDays($days));
		                               })
		                               ->with('assignedUser')
		                               ->orderBy('updated_at', 'desc')
		                               ->get();
	}
}
