<?php
namespace App\Repositories\Eloquent;
use App\Models\CRM\Customer;
use App\Repositories\Interfaces\CustomerRepositoryInterface;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
class CustomerRepository implements CustomerRepositoryInterface
{
	public function all(): LengthAwarePaginator|array {
		return Customer::with('region')
		               ->latest()
		               ->paginate(20);
	}
	public function getAllWithRelations(): LengthAwarePaginator {
		return Customer::query()
		               ->with([
			               'region',
			               'customerType',
			               'assignedUser',
			               'creator',
			               'latestLead' => function ($query) {
				               $query->with('source', 'status', 'assignedUser');
			               },
			               'latestOrder',
			               'latestPayment'
		               ])
		               ->latest('id')
		               ->paginate(20);
	}
	public function find($id)
	{
		return Customer::with('region', 'leads')
		               ->findOrFail($id);
	}
	public function findWithDetails($id)
	{
		return Customer::with([
			'region',
			'customerType',
			'assignedUser',
			'creator',
			'leads' => function ($query) {
				$query->with('source', 'status', 'assignedUser')->latest();
			},
			'orders' => function ($query) {
				$query->with('items.product')->latest();
			},
			'payments' => function ($query) {
				$query->with('method')->latest();
			},
			'tags',
			'customerRatings.ratingType',
			'interactions',
			'memberships.package',
			'debts',
			'tasks' => function ($query) {
				$query->whereIn('status', ['pending', 'in_progress'])->latest();
			},
		])->findOrFail($id);
	}
	public function search(array $filters = []): LengthAwarePaginator
	{
		$query = Customer::query()->with([
			'region',
			'customerType',
			'assignedUser',
			'creator',
			'latestLead.source',
			'latestOrder',
		]);
		// Tìm kiếm theo tên, số điện thoại, email
		if (!empty($filters['search'])) {
			$search = $filters['search'];
			$query->where(function ($q) use ($search) {
				$q->where('name', 'like', "%{$search}%")
				  ->orWhere('phone', 'like', "%{$search}%")
				  ->orWhere('email', 'like', "%{$search}%")
				  ->orWhere('facebook_name', 'like', "%{$search}%")
				  ->orWhere('nickname', 'like', "%{$search}%");
			});
		}
		// Lọc theo khu vực
		if (!empty($filters['region_id'])) {
			$query->where('region_id', $filters['region_id']);
		}
		// Lọc theo loại khách
		if (!empty($filters['customer_type_id'])) {
			$query->where('customer_type_id', $filters['customer_type_id']);
		}
		// Lọc theo trạng thái
		if (!empty($filters['customer_status'])) {
			$query->where('customer_status', $filters['customer_status']);
		}
		// Lọc theo người phụ trách
		if (!empty($filters['owner_id'])) {
			$query->where('owner_id', $filters['owner_id']);
		}
		// Lọc theo khách hàng tiềm năng
		if (isset($filters['is_potential'])) {
			$query->where('is_potential', $filters['is_potential']);
		}
		return $query->latest('id')->paginate($filters['per_page'] ?? 20);
	}
	public function create(array $data)
	{
		return Customer::create($data);
	}
	public function update($id, array $data): Customer|array {
		$customer = Customer::findOrFail($id);
		$customer->update($data);
		return $customer;
	}
	public function delete($id): int {
		return Customer::destroy($id);
	}
	public function count(): int {
		return DB::table('crm_customers')->count();
	}
	public function countPurchased(): int {
		return DB::table('crm_customers')
		         ->join('crm_leads', 'crm_leads.customer_id', '=', 'crm_customers.id')
		         ->join('crm_orders', 'crm_orders.lead_id', '=', 'crm_leads.id')
		         ->distinct('crm_customers.id')
		         ->count('crm_customers.id');
	}
}
