<?php
namespace App\Models\CRM;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasOneThrough;
class Order extends Model
{
	use HasFactory;
	protected $table = 'crm_orders';
	protected $fillable = [
		'lead_id',
		'order_code',
		'order_date',
		'status',
		'warehouse_id',
		'total_amount',
		'created_by',
		'approved_by',
		'approved_at',
		'payment_recorded',
        'current_department',
        'current_status_type_id',
        'estimated_delivery',
	];
    protected $casts = [
        'order_date'        => 'date',
        'approved_at'       => 'datetime',
        'estimated_delivery'=> 'datetime',
    ];
	public function lead(): BelongsTo {
		return $this->belongsTo(Lead::class, 'lead_id');
	}
	public function customer(): HasOneThrough|Order {
		return $this->hasOneThrough(
			Customer::class,
			Lead::class,
			'id',          // Lead.id
			'id',          // Customer.id
			'lead_id',     // Order.lead_id
			'customer_id', // Lead.customer_id
		);
	}
	public function items(): HasMany|Order {
		return $this->hasMany(OrderItem::class);
	}
	public function approvals(): HasMany|Order {
		return $this->hasMany(OrderApproval::class);
	}
	public function warehouse(): BelongsTo {
		return $this->belongsTo(Warehouse::class);
	}
	public function creator(): BelongsTo {
		return $this->belongsTo(User::class, 'created_by');
	}
	public function approver(): BelongsTo {
		return $this->belongsTo(User::class, 'approved_by');
	}
	public function payments(): HasMany|Order {
		return $this->hasMany(Payment::class);
	}
	public function latestPayment()
	{
		return $this->hasOne(Payment::class)->latestOfMany('payment_date');
	}
	public function latestApproval()
	{
		return $this->hasOne(OrderApproval::class)->latestOfMany('approved_at');
	}
	public function latestApprovalLevel(): ?string
	{
		return $this->approvals()
		            ->where('status', 'approved')
		            ->orderBy('approved_at', 'desc')
		            ->value('level');
	}
	// Customer Debt
	public function debt(): HasOne|Order {
		return $this->hasOne(CustomerDebt::class);
	}
	// Sales Tasks
	public function tasks(): HasMany|Order {
		return $this->hasMany(SalesTask::class);
	}
	public function currentStatusType(): BelongsTo
	{
		return $this->belongsTo(OrderStatusType::class, 'current_status_type_id');
	}
    public function getStatusColor(): string
    {
        return match($this->current_department) {
            'sales' => 'secondary',
            'ketoan' => 'info',
            'duyet1', 'duyet2' => 'warning',
            'kho' => 'primary',
            'completed' => 'success',
            default => 'dark',
        };
    }

    public function getDepartmentName(): string
    {
        return match($this->current_department) {
            'sales' => 'Sales',
            'ketoan' => 'Kế toán',
            'duyet1' => 'Duyệt cấp 1',
            'duyet2' => 'Giám đốc',
            'kho' => 'Kho',
            'completed' => 'Hoàn tất',
            default => 'N/A',
        };
    }
}
