<?php
namespace App\Models\CRM;
use App\Models\Region;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
class Customer extends Model
{
	use HasFactory;
	protected $table = 'crm_customers';
	protected $fillable = [
		'name',
		'phone',
		'facebook_name',
		'facebook_link',
		'zalo_id',
		'customer_type_id',
		'region_id',
		'email',
		'address',
		'nickname',
		'group_note',
		'owner_id',
		'customer_status',
		'is_potential',
		'converted_to_member_at',
		'created_by',
	];
	protected $casts = [
		'is_potential' => 'boolean',
		'converted_to_member_at' => 'datetime',
	];
	// Vùng miền
	public function region(): BelongsTo {
		return $this->belongsTo(Region::class);
	}
	// Loại khách
	public function customerType(): BelongsTo {
		return $this->belongsTo(CustomerType::class, 'customer_type_id');
	}
	// Người phụ trách
	public function assignedUser(): BelongsTo {
		return $this->belongsTo(User::class, 'owner_id');
	}
	// Người tạo
	public function creator(): BelongsTo {
		return $this->belongsTo(User::class, 'created_by');
	}
	// Lead
	public function leads(): HasMany {
		return $this->hasMany(Lead::class);
	}
	public function latestLead()
	{
		return $this->hasOne(Lead::class)->latestOfMany('created_at');
	}
	// Orders thông qua Leads
	public function orders(): HasManyThrough {
		return $this->hasManyThrough(
			Order::class,
			Lead::class,
			'customer_id', // Lead.customer_id
			'lead_id',     // Order.lead_id
			'id',          // Customer.id
			'id'           // Lead.id
		);
	}
	// Order mới nhất
	public function latestOrder()
	{
		return $this->hasOneThrough(
			Order::class,
			Lead::class,
			'customer_id',
			'lead_id',
			'id',
			'id'
		)->latest('order_date');
	}
	// Payment thông qua Orders → Leads
	public function payments(): HasManyThrough {
		return $this->hasManyThrough(
			Payment::class,
			Order::class,
			'lead_id',  // Order.lead_id → dẫn về Lead
			'order_id', // Payment.order_id
			'id',       // Customer.id
			'id'        // Lead.id
		);
	}
	public function latestPayment()
	{
		return $this->hasOneThrough(
			Payment::class,
			Order::class,
			'lead_id',   // Order.lead_id → dẫn về lead
			'order_id',  // Payment.order_id
			'id',        // Customer.id
			'id'         // Lead.id
		)->latestOfMany('payment_date');
	}
	// Tags (nhiều-nhiều) - FIXED
	public function tags(): BelongsToMany
	{
		return $this->belongsToMany(
			Tag::class,
			'crm_customer_tags',
			'customer_id',
			'tag_id'
		)->withPivot('tagged_by', 'tagged_at');
	}
	// Customer Ratings
	public function customerRatings(): HasMany
	{
		return $this->hasMany(CustomerRating::class);
	}
	public function latestRating()
	{
		return $this->hasOne(CustomerRating::class)->latestOfMany('created_at');
	}
	// Customer Interactions
	public function interactions(): HasMany
	{
		return $this->hasMany(CustomerInteraction::class);
	}
	// Memberships
	public function memberships(): HasMany
	{
		return $this->hasMany(Membership::class);
	}
	public function activeMembership()
	{
		return $this->hasOne(Membership::class)
		            ->where('status', 'active')
		            ->where('end_date', '>=', now());
	}
	// Customer Debts
	public function debts(): HasMany
	{
		return $this->hasMany(CustomerDebt::class);
	}
	public function unpaidDebts()
	{
		return $this->hasMany(CustomerDebt::class)
		            ->whereIn('status', ['unpaid', 'partial', 'overdue']);
	}
	// Sales Tasks
	public function tasks(): HasMany
	{
		return $this->hasMany(SalesTask::class);
	}
	public function pendingTasks()
	{
		return $this->hasMany(SalesTask::class)
		            ->whereIn('status', ['pending', 'in_progress']);
	}
}