<?php

namespace App\Http\Requests;
use Illuminate\Contracts\Validation\ValidationRule;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class ProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
	    return auth()->check() && auth()->user()->can('products.manage');
    }
    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
	    $productId = $this->route('product')?->id;

	    return [
		    'name' => ['required','string','max:255', Rule::unique('crm_product_catalog')->ignore($productId)],
		    'sku' => ['required','string','max:100', Rule::unique('crm_product_catalog')->ignore($productId)],
		    'unit' => 'nullable|string|max:50',
		    'category_id' => 'nullable|exists:crm_product_categories,id',
		    'price_agent' => ['required','numeric','min:0'],
		    'price_retail' => ['required','numeric','min:0'],
		    'quantity' => ['nullable','integer','min:0'],
		    'warehouse_id' => ['nullable','exists:crm_warehouses,id'],
		    'main_image_id' => ['nullable','exists:media_files,id'],
		    'gallery_ids' => ['nullable','string'],
	    ];
    }

	/**
	 * @param Validator $validator
	 *
	 * @return void
	 */
	public function withValidator(Validator $validator): void {
		$validator->after(function ($validator) {
			$qty = $this->input('quantity');
			if ($qty !== null) {
				$qty = (int) $qty;
			}
			$warehouse = $this->input('warehouse_id');
			if ($qty !== null && $qty > 0 && empty($warehouse)) {
				$validator->errors()->add('warehouse_id', 'Phải chọn kho nếu nhập số lượng ban đầu.');
			}
		});
	}
	/**
	 * @return string[]
	 */
	public function messages(): array {
		return [
			'price_agent.min' => 'Giá đại lý phải >= 0.',
			'price_retail.min' => 'Giá bán lẻ phải >= 0.',
			'quantity.min' => 'Số lượng phải >= 0.',
			'warehouse_id.exists' => 'Kho không hợp lệ.',
		];
	}
}
